function [ ] = LUAM3_save_results(LU_endogenous, LU_exogenous, GISheader, output_path, years, pixel_size, c, LURNZDATA, ForestAge_endogenous, ScrubAge_endogenous, LUAM_PDFS, LUAM_ASCI, ALL_YEARS )
%
% Land Use Allocation Module - function 3 (LUAM3)
%
% This file saves the estimated map of land use in pdf and asc form.
%

% Code by Simon Anastasiadis : 2011-11-03

%% Parameters

% module output folder
module_folder = '\Results 2 Land Use Allocation Module';
% GIS output folder
GIS_folder    = '\Results 2 Land Use Allocation Module\GIS data';
% PDf map output folder
pdfmap_folder = '\Results 2 Land Use Allocation Module\LandUse Maps';

%% Set folder to store results

% make module folder
if ~exist([output_path,module_folder],'dir')
    mkdir([output_path,module_folder]);
end

% make GIS output folder
if ~exist([output_path,GIS_folder],'dir')
    mkdir([output_path,GIS_folder]);
end

% make PDF map folder
if ~exist([output_path,pdfmap_folder],'dir')
    mkdir([output_path,pdfmap_folder]);
end

%% Call function to make and save transition tables

SaveTransitionTables(years, LU_endogenous, output_path, module_folder, pixel_size)

%% Save ASCI maps
% This code only saves asci maps for the last simulation year.
% ASCI maps of other years can be produced using the SaveAsciMaps function
% and the data saved in the Matlab output file.

% if saving asci maps
if LUAM_ASCI
    % loop through years
    for ii = 1:length(years)
        % if saving all years // always saving final year
        if ALL_YEARS || ii == length(years)

            % put map data together for last simulation year
            MapData = [LU_endogenous(:,1), LU_endogenous(:,ii+1)];
            MapData = [MapData ; LU_exogenous];
            
            % save as asci map
            SaveAsciMaps(MapData, GISheader, output_path, GIS_folder, years(ii) );
            
            % Remove MapData to save space
            clear('MapData')
        end
    end
end

%% Save Land Use Maps
% This code saves pdf maps of Land Use for each year in the simulation.
% This is a lengthy process. Some users may wish to specify that LURNZ
% should only save specific pdf maps - this is yet to be coded.

% if saving pdf maps
if LUAM_PDFS
    % loop through years
    for ii = 1:length(years)
        % if saving all years // always saving final year
        if ALL_YEARS || ii == length(years)
            
            % file name and path
            mappath  = [output_path, pdfmap_folder, '\Landuse map ', ...
                        num2str(years(ii)), '.pdf'];
            % map title
            maptitle = ['Land Use, year ', num2str(years(ii))];
            
            % assemble map grid
            mapfill = [[LU_endogenous(:,1), LU_endogenous(:,ii+1)];...
                       [LU_exogenous(:,1) , LU_exogenous(:,2)] ];
            % save as pdf map
            SaveLandUseMaps(mapfill, GISheader, mappath, maptitle);
        end
    end
    
    % Remove mapgrid to save space
    clear('mapgrid')
end

%% Save Land Use Transition Maps
% This code only saves pdf maps of Land Use Transitions between the first
% and last simulation years, as this is the time factor over which change
% is most visable and generally of greatest interest.
% PDF maps of Land Use Transitions between other years can be produced
% using the SaveLandUseTransitions function and the data saved in the
% Matlab output file.

% if saving pdf maps
if LUAM_PDFS
    % Select map data for start year (land use change FROM this year)
    FromYear = years(1);
    % Map = [sorter, endogenous Land Use]
    FromMap  = [LU_endogenous(:,1),LU_endogenous(:,2)];
    % Include exogenous land use
    FromMap = [FromMap ; LU_exogenous];
    
    % Select map data for end year (land use change TO this year)
    ToYear   = years(end);
    % Map = [sorter, endogenous Land Use]
    ToMap    = [LU_endogenous(:,1),LU_endogenous(:,end)];
    % Include exogenous land use
    ToMap = [ToMap ; LU_exogenous];
    
    % save as pdf map
    SaveLandUseTransitions(FromMap, ToMap, GISheader, output_path, pdfmap_folder, FromYear, ToYear)
    
    % Remove maps to save space
    clear('FromMap')
    clear('ToMap')
end

%% Save summary tables by Territorial Authority & Regional Council
% This code only saves TA & RC summary tables for the last simulation year,
% as this is anticipated to be the year of greatest interest.
% TA & RC level summary tables for other years can be produced using the
% SaveTASummaryTables and SaveRCSummaryTables functions.

% land use map = endogenous land
LUmap = [LU_endogenous(:,1),LU_endogenous(:,end)];
% land use map + exogenous land
LUmap = [LUmap ; LU_exogenous];

% sort to match LURNZDATA
LUmap = sortrows(LUmap, 1);

% save summary tables
SaveTASummaryTables(LUmap, LURNZDATA(:,c.ta), years(end), output_path, module_folder, pixel_size );
SaveRCSummaryTables(LUmap, LURNZDATA(:,c.rc), years(end), output_path, module_folder, pixel_size );

%%  Save Matlab output

% clear excess variables
clear('FromYear', 'GIS_folder', 'LUmap', 'ToYear', ...
      'ii', 'mapfill', 'mappath', 'maptitle', 'pdfmap_folder');

% save output
if pixel_size == 1
    save([output_path,module_folder,'\LUAM_output_data'],'-v7.3')
elseif pixel_size == 25
    save([output_path,module_folder,'\LUAM_output_data'])
end

%% Inform User saving land use change is complete

msg = sprintf(' - - LUAM3 : save data complete');
disp(msg)

end


%% Subfunction : Make and Save transition tables

function SaveTransitionTables(years, LU_endogenous, output_path, module_folder, pixel_size)


%% Transition Tables - Setting up

% Column names
TTable_cols = {'Land out of Dairy'   , 'Land out of Sheep & Beef', ...
               'Land out of Forestry', 'Land out of Scrub'};
% Row names
TTable_rows = {'Land into Dairy'    ; 'Land into Sheep & Beef';...
               'Land into Forestry' ; 'Land into Scrub'};

% Pixels file name
pixel_file = '\Transition Table - pixel changes.xls';
% Pixels subtitle
pixels_subtitle = {'Number of pixels'};

% Hectares file name
hectare_file = '\Transition Table - hectare changes.xls';
% Hectares subtitle
hectare_subtitle = {'Area in hectares'};

%% Transition Tables - Construct Tables

% empty transition tables
% rows    = 4 land uses (dairy, sheep & beef, forest, scrub)
% columns = 4 land uses (dairy, sheep & beef, forest, scrub)
% depth   = simulation years
TransitionTable = zeros(4,4,length(years)-1);

% fill transition table
% loop through simulation years
for yy = 2:length(years)
    % TransitionTable(ii,jj,yy) = 
    % 'number of pixels of type jj that became type ii in year yy'
    
    %loop through post land uses
    for ii = 1:4
        % loop through pre land uses
        for jj = 1:4
            % fill table
            change = LU_endogenous(:,yy)==jj & LU_endogenous(:,yy+1)==ii;
            TransitionTable(ii,jj,yy-1) = sum(change);
        end
    end
end

%% Transition Tables - Export to Excel
% it is faster to loop through each file separately than both files
% together

% silence warnings
warning('OFF','MATLAB:xlswrite:AddSheet')

% loop through Transition table layers
for yy = 1:size(TransitionTable,3)
    % sheet name
    sheet = ['from ',num2str(years(yy)),' to ',num2str(years(yy+1))];

    % Write to Hectares file
    file = [output_path,module_folder,hectare_file];
    xlswrite(file,hectare_subtitle,sheet,'A1');
    xlswrite(file,TTable_cols,sheet,'B1');
    xlswrite(file,TTable_rows,sheet,'A2');
    xlswrite(file,TransitionTable(:,:,yy)*pixel_size,sheet,'B2')
end

% delete the empty sheets
deleteEmptyExcelSheets([output_path,module_folder,hectare_file]);

% loop through Transition table layers
for yy = 1:size(TransitionTable,3)
    % sheet name
    sheet = ['from ',num2str(years(yy)),' to ',num2str(years(yy+1))];
    
    % Write to Pixel file
    file = [output_path,module_folder,pixel_file];
    xlswrite(file,pixels_subtitle,sheet,'A1');
    xlswrite(file,TTable_cols,sheet,'B1');
    xlswrite(file,TTable_rows,sheet,'A2');
    xlswrite(file,TransitionTable(:,:,yy),sheet,'B2')
end

% delete the empty sheets
deleteEmptyExcelSheets([output_path,module_folder,pixel_file]);

% permit warnings
warning('ON','MATLAB:xlswrite:AddSheet')

end

%% Subfunction : Save ASCI maps

function SaveAsciMaps(MapData, GISheader, output_path, GIS_folder, year )

% unpack GISheader
% Map_GISHeader = [Longitude(ncols);Latitude(nrows);Xll;Yll;CellSize;No_Value]
Map_Longitude = GISheader(1);
Map_Latitude  = GISheader(2);
Map_No_Value  = GISheader(6);

% construct empty map
Map = Map_No_Value * ones(Map_Latitude, Map_Longitude);

% fill map
Map(MapData(:,1)) = MapData(:,2);

% Save
file = [output_path,GIS_folder,'\LandUseMap_Year_',num2str(year),'.asc'];
SaveAsciiRaster(file, Map, GISheader);

end

%% Subfunction : Save Land Use Transition Maps as PDFs (1 of 2)

function SaveLandUseTransitions(FromMap, ToMap, GISheader, output_path, pdfmap_folder, FromYear, ToYear)

% mark entries that are unchanged
% FromMap and ToMap should be sorted the same by construction.
iNoChange = FromMap(:,2) == ToMap(:,2);

% Unchanged land has land use zero
FromMap(iNoChange,2) = 0;
ToMap(iNoChange,2) = 0;

% Construct empty map grids
Map_Longitude = GISheader(1);
Map_Latitude  = GISheader(2);
FromGrid = -9999 * ones(Map_Latitude, Map_Longitude);
ToGrid   = -9999 * ones(Map_Latitude, Map_Longitude);

% fill map grids
FromGrid(FromMap(:,1)) = FromMap(:,2);
ToGrid(ToMap(:,1)) = ToMap(:,2);

% map titles
FromTitle = ['Transition - original land use ', ...
            num2str(FromYear), '-', num2str(ToYear)];
ToTitle   = ['Transition - final land use ', ...
            num2str(FromYear), '-', num2str(ToYear)];
% file names and paths
FromFile = [output_path, pdfmap_folder, '\', FromTitle, '.pdf'];
ToFile   = [output_path, pdfmap_folder, '\', ToTitle  , '.pdf'];

% Call second subfunction for graphing and saving
PlotLandUseTransitions(FromGrid, FromTitle, FromFile);
PlotLandUseTransitions(ToGrid,   ToTitle,   ToFile);

end

%% Subfunction : Save Land Use Transition Maps as PDFs (2 of 2)

function PlotLandUseTransitions(mapgrid, maptitle, mappath)

% we will construct map as a [Red Green Blue] colour layer
% first construct all 3 colour layers separately
R = ones(size(mapgrid));
G = ones(size(mapgrid));
B = ones(size(mapgrid));

% fill map with colours [R,G,B]
% dairy          = 1 = red
% sheep & beef   = 2 = yellow
% forest         = 3 = green
% scrub          = 4 = blue
% unchanged land = 0 = grey
R(mapgrid==1) = 1;    G(mapgrid==1) = 0;    B(mapgrid==1) = 0;
R(mapgrid==2) = 1;    G(mapgrid==2) = 1;    B(mapgrid==2) = 0;
R(mapgrid==3) = 0;    G(mapgrid==3) = 1;    B(mapgrid==3) = 0;
R(mapgrid==4) = 0;    G(mapgrid==4) = 0;    B(mapgrid==4) = 1;
R(mapgrid==0) = 0.66; G(mapgrid==0) = 0.66; B(mapgrid==0) = 0.66;
% We choose the grey [169/256 169/256 169/256] as this ensures the highest
% contrast ratio of the grey vs. each of the red, yellow, green and blue.

% Combine colour layers to form a colour map
im = zeros([size(mapgrid),3]);
im(:,:,1) = R;
im(:,:,2) = G;
im(:,:,3) = B;

% Special mapping commands
scrsz = get(0,'ScreenSize');
set(gcf,'Position',[30 40 scrsz(3)/2 scrsz(4)-140]);
image(im);shading flat; axis equal tight off;

% Legend
hold all;
% legend colours
h=bar(zeros(4,4));
set(h(1),'facecolor',[1 0 0])
set(h(2),'facecolor',[1 1 0])
set(h(3),'facecolor',[0 1 0])
set(h(4),'facecolor',[0 0 1])
% legend names
legend('Dairy','Sheep & Beef','Forestry','Scrub','Location','NorthWest');
legend('boxoff');
hold off;

% Give title
title(maptitle);

% save to pdf
save2pdf(mappath);
close;

end

%% Subfunction : Save Land Use Map as PDFs

function SaveLandUseMaps(mapfill, GISheader, mappath, maptitle)

% Construct empty map grids
Map_Longitude = GISheader(1);
Map_Latitude  = GISheader(2);
mapgrid = -9999 * ones(Map_Latitude, Map_Longitude);

% fill map grids
mapgrid(mapfill(:,1)) = mapfill(:,2);

% we will construct map as a [Red Green Blue] colour layer
% first construct all 3 colour layers separately
R = ones(size(mapgrid));
G = ones(size(mapgrid));
B = ones(size(mapgrid));

% fill map with colours [R,G,B]
% dairy                   =  1 = red
% sheep & beef            =  2 = yellow
% plantation forest       =  3 = green
% scrub                   =  4 = blue
% horticulture            =  5 = magenta
% Non-productive land     =  6 = black
% Urban                   =  7 = grey
% other animals lifestyle =  8 = dull pink
% indigenous forest       =  9 = dark green
% pasture on public land  = 10 = orange
% DoC and public land     = 11 = black
R(mapgrid== 1) = 1;   G(mapgrid== 1) = 0;      B(mapgrid== 1) = 0;
R(mapgrid== 2) = 1;   G(mapgrid== 2) = 1;      B(mapgrid== 2) = 0;
R(mapgrid== 3) = 0;   G(mapgrid== 3) = 1;      B(mapgrid== 3) = 0;
R(mapgrid== 4) = 0;   G(mapgrid== 4) = 0;      B(mapgrid== 4) = 1;
R(mapgrid== 5) = 1;   G(mapgrid== 5) = 0;      B(mapgrid== 5) = 1;
R(mapgrid== 6) = 0;   G(mapgrid== 6) = 0;      B(mapgrid== 6) = 0;
R(mapgrid== 7) = 0.5; G(mapgrid== 7) = 0.5;    B(mapgrid== 7) = 0.5;
R(mapgrid== 8) = 1;   G(mapgrid== 8) = 0.75;   B(mapgrid== 8) = 0.9730;
R(mapgrid== 9) = 0;   G(mapgrid== 9) = 0.3906; B(mapgrid== 9) = 0;
R(mapgrid==10) = 1;   G(mapgrid==10) = 0.6445; B(mapgrid==10) = 0;
R(mapgrid==11) = 0;   G(mapgrid==11) = 0;      B(mapgrid==11) = 0;

% Combine colour layers to form a colour map
im = zeros([size(mapgrid),3]);
im(:,:,1) = R;
im(:,:,2) = G;
im(:,:,3) = B;

% Special mapping commands
scrsz = get(0,'ScreenSize');
set(gcf,'Position',[30 40 scrsz(3)/2 scrsz(4)-140]);
image(im);shading flat; axis equal tight off;

% Legend
hold all;
% legend colours
h=bar(zeros(11,11));
set(h(1),'facecolor',[1 0 0])
set(h(2),'facecolor',[1 1 0])
set(h(3),'facecolor',[0 1 0])
set(h(4),'facecolor',[0 0 1])
set(h(5),'facecolor',[1 0 1])
set(h(6),'facecolor',[0 0 0])
set(h(7),'facecolor',[128/256 128/256 128/256])
set(h(8),'facecolor',[1 192/256 203/256])
set(h(9),'facecolor',[0 100/256 0])
set(h(10),'facecolor',[1 165/256 0])
set(h(11),'facecolor',[0 0 0])
% legend names
legend('Dairy', 'Sheep & Beef', 'Forestry', 'Scrub', 'Horticulture', ...
       'Non-productive', 'Urban', 'Lifestyle/Other animal', ...
       'Indigenous forest', 'Public pasture', 'DoC/Public land', ...
       'Location','NorthWest');
legend('boxoff');
hold off;

% Give title
title(maptitle);

% save to pdf
save2pdf(mappath);
close;

end

%% Subfunction : Save Territorial Authority Summary Tables

function SaveTASummaryTables(LUmap, column, year, output_path, module_folder, pixel_size )

% add TA identifiers onto LUmap
LUmap(:,3) = column;

% identify TAs
tas = unique(LUmap(:,3));
tas = tas(~isnan(tas));
tas = sort(tas);

%% Determine areas

% count pixels
tas_areas = zeros(length(tas), 11);
% fill tas_areas
for ii = 1:length(tas)
    tas_areas(ii, 1) = sum(LUmap(:,3)==tas(ii) & LUmap(:,2)== 1);
    tas_areas(ii, 2) = sum(LUmap(:,3)==tas(ii) & LUmap(:,2)== 2);
    tas_areas(ii, 3) = sum(LUmap(:,3)==tas(ii) & LUmap(:,2)== 3);
    tas_areas(ii, 4) = sum(LUmap(:,3)==tas(ii) & LUmap(:,2)== 4);
    tas_areas(ii, 5) = sum(LUmap(:,3)==tas(ii) & LUmap(:,2)== 5);
    tas_areas(ii, 6) = sum(LUmap(:,3)==tas(ii) & LUmap(:,2)== 6);
    tas_areas(ii, 7) = sum(LUmap(:,3)==tas(ii) & LUmap(:,2)== 7);
    tas_areas(ii, 8) = sum(LUmap(:,3)==tas(ii) & LUmap(:,2)== 8);
    tas_areas(ii, 9) = sum(LUmap(:,3)==tas(ii) & LUmap(:,2)== 9);
    tas_areas(ii,10) = sum(LUmap(:,3)==tas(ii) & LUmap(:,2)==10);
    tas_areas(ii,11) = sum(LUmap(:,3)==tas(ii) & LUmap(:,2)==11);
end

% scale by pixel size
tas_areas = tas_areas * pixel_size;

%% Save

% labels
row_headers = {'TA code','TA';1,'Far North District';2,'Whangarei District';3,'Kaipara District';4,'Rodney District';5,'North Shore City';6,'Waitakere City';7,'Auckland City';8,'Manukau City';9,'Papakura District';10,'Franklin District';11,'Thames-Coromandel District';12,'Hauraki District';13,'Waikato District';15,'Matamata-Piako District';16,'Hamilton City';17,'Waipa District';18,'Otorohanga District';19,'South Waikato District';20,'Waitomo District';21,'Taupo District';22,'Western Bay of Plenty District';23,'Tauranga District';24,'Rotorua District';25,'Kawerau District';26,'Whakatane District';27,'Opotiki District';28,'Gisborne District';29,'Wairoa District';30,'Napier City';31,'Hastings District';32,'Central Hawkes Bay District';33,'New Plymouth District';34,'Stratford District';35,'South Taranaki District';36,'Ruapehu District';37,'Wanganui District';38,'Rangitikei District';39,'Manawatu District';40,'Palmerston North City';41,'Tararua District';42,'Horowhenua District';43,'Kapiti Coast';44,'Porirua City';45,'Hutt District';46,'Lower Hutt City';47,'Wellington City';48,'Masterton District';49,'Carterton District';50,'South Wairarapa District';51,'Tasman District';52,'Nelson City';53,'Marlborough District';54,'Kaikoura District';55,'Buller District';56,'Grey District';57,'Westland District';58,'Hurunui District';59,'Waimakariri District';60,'Christchurch City';61,'Banks Peninsula District';62,'Selwyn District';63,'Ashburton District';64,'Timaru District';65,'Mackenzie District';66,'Waimate District';68,'Waitaki District';69,'Central Otago District';70,'Queenstown-Lakes District';71,'Dunedin City';72,'Clutha District';73,'Southland District';74,'Gore District';75,'Invercargill City';999,'Area not in a TA';};
col_headers = {'Land use (hectares)',[],[],[],[],[],[],[],[],[],[];'Dairy','Sheep & Beef','Forestry','Scrub','Horticulture','Non-productive','Urban','Other animal and lifestyle','Indigenous Forest','Pasture on public land','DoC and public land';};

file = [output_path,module_folder,'\TA land use table ',num2str(year),'.xls'];

xlswrite(file,year,'Sheet1','A1');
xlswrite(file,row_headers,'Sheet1','A2');
xlswrite(file,col_headers,'Sheet1','C1');
xlswrite(file,tas_areas(1:end-1,:),'Sheet1','C3');

end

%% Subfunction : Save Regional Council Summary Tables

function SaveRCSummaryTables(LUmap, column, year, output_path, module_folder, pixel_size )

% add TA identifiers onto LUmap
LUmap(:,3) = column;

% identify RCs
rcs = [1:9, 12:18];

%% Determine areas

% count pixels
rcs_areas = zeros(length(rcs), 11);
% fill tas_areas
for ii = 1:length(rcs)
    rcs_areas(ii, 1) = sum(LUmap(:,3)==rcs(ii) & LUmap(:,2)== 1);
    rcs_areas(ii, 2) = sum(LUmap(:,3)==rcs(ii) & LUmap(:,2)== 2);
    rcs_areas(ii, 3) = sum(LUmap(:,3)==rcs(ii) & LUmap(:,2)== 3);
    rcs_areas(ii, 4) = sum(LUmap(:,3)==rcs(ii) & LUmap(:,2)== 4);
    rcs_areas(ii, 5) = sum(LUmap(:,3)==rcs(ii) & LUmap(:,2)== 5);
    rcs_areas(ii, 6) = sum(LUmap(:,3)==rcs(ii) & LUmap(:,2)== 6);
    rcs_areas(ii, 7) = sum(LUmap(:,3)==rcs(ii) & LUmap(:,2)== 7);
    rcs_areas(ii, 8) = sum(LUmap(:,3)==rcs(ii) & LUmap(:,2)== 8);
    rcs_areas(ii, 9) = sum(LUmap(:,3)==rcs(ii) & LUmap(:,2)== 9);
    rcs_areas(ii,10) = sum(LUmap(:,3)==rcs(ii) & LUmap(:,2)==10);
    rcs_areas(ii,11) = sum(LUmap(:,3)==rcs(ii) & LUmap(:,2)==11);
end

% scale by pixel size
rcs_areas = rcs_areas * pixel_size;

%% Save

% labels
row_headers = {'RC code','RC';1,'Northland';2,'Auckland';3,'Waikato';4,'Bay of Plenty';5,'Gisborne';6,'Hawkes Bay';7,'Taranaki';8,'Manawatu-Wanganui';9,'Wellington';12,'West Coast';13,'Canterbury';14,'Otago';15,'Southland';16,'Tasman';17,'Nelson';18,'Marlborough';};
col_headers = {'Land use (hectares)',[],[],[],[],[],[],[],[],[],[];'Dairy','Sheep & Beef','Forestry','Scrub','Horticulture','Non-productive','Urban','Other animal and lifestyle','Indigenous Forest','Pasture on public land','DoC and public land';};

file = [output_path,module_folder,'\RC land use table ',num2str(year),'.xls'];

xlswrite(file,year,'Sheet1','A1');
xlswrite(file,row_headers,'Sheet1','A2');
xlswrite(file,col_headers,'Sheet1','C1');
xlswrite(file,rcs_areas,'Sheet1','C3');

end


